import xarray as xr
import matplotlib.pyplot as plt
import cartopy.crs as ccrs
import cartopy.feature as cfeature
from matplotlib.colors import BoundaryNorm, ListedColormap
from matplotlib.cm import ScalarMappable
import numpy as np

# Load dataset
ds = xr.open_dataset("tx_ens_mean_0.25deg_reg_v31.0e.nc")

# Select 26 June 2019 temperature data (remove time dimension)
tx_26jun2019 = ds.sel(time="2019-06-26")['tx'].squeeze()

# Compute June climatology (1981-2010 average of June means)
clim_start, clim_end = "1981-01-01", "2010-12-31"
june_clim = ds.sel(time=slice(clim_start, clim_end))
june_clim = june_clim.sel(time=june_clim['time.month'] == 6)['tx']
june_clim = june_clim.groupby("time.year").mean("time").mean("year")

# Compute anomaly for 26 June 2019
anomaly_26jun2019 = tx_26jun2019 - june_clim

# Plotting function
def plot_map(ax, data, cmap, norm, title=None):
    ax.set_extent([-25, 45, 34, 72], crs=ccrs.PlateCarree())
    ax.add_feature(cfeature.COASTLINE, linewidth=0.5)
    ax.add_feature(cfeature.BORDERS, linewidth=0.4)
    ax.add_feature(cfeature.LAND, facecolor='lightgrey')

    im = ax.pcolormesh(
        data['longitude'], data['latitude'], data,
        cmap=cmap, shading='auto', norm=norm,
        transform=ccrs.PlateCarree()
    )

    if title:
        ax.set_title(title, fontsize=14, fontweight='bold', pad=12)

    return im

# Create figure and axes
fig, axes = plt.subplots(1, 2, figsize=(11, 6), subplot_kw={'projection': ccrs.PlateCarree()})

# Plot absolute temperature on 26 June 2019 with continuous norm
im1 = plot_map(
    axes[0], tx_26jun2019,
    cmap=plt.get_cmap('magma_r'),
    norm=plt.Normalize(vmin=15, vmax=43),
    title="26 June 2019 - Temp (°C)"
)

# Define finer uneven boundaries for anomaly for smooth and proportional coloring
bounds = np.array([-5, -4, -3, -2, -1, -0.5, 0, 0.5, 1, 2, 3, 5, 7, 10, 12, 15])

# Number of intervals
ncolors = len(bounds) - 1

# Discretize colormap into ncolors intervals
cmap_base = plt.get_cmap('RdBu_r')
colors = cmap_base(np.linspace(0, 1, ncolors))
cmap = ListedColormap(colors)

# Create BoundaryNorm with these boundaries
norm = BoundaryNorm(bounds, ncolors)

# Plot anomaly
im2 = plot_map(
    axes[1], anomaly_26jun2019,
    cmap=cmap,
    norm=norm,
    title="26 June 2019 - Temp Anomaly (°C)"
)

# Colorbar for absolute temperature
cbar1 = fig.colorbar(im1, ax=axes[0], orientation='horizontal', pad=0.01, fraction=0.05, aspect=30)
cbar1.set_label("Temperature (°C)", fontsize=12)

# Create ScalarMappable for anomaly colorbar
sm = ScalarMappable(norm=norm, cmap=cmap)
sm.set_array([])

# Place ticks at midpoints of intervals for better labeling
tick_locs = (bounds[:-1] + bounds[1:]) / 2

# Pretty labels: just the midpoint rounded, with a special bold zero label
tick_labels = [f"{loc:.1f}" for loc in tick_locs]

# Find zero index for bold label styling later
try:
    zero_idx = list(np.round(tick_locs, 1)).index(0.0)
except ValueError:
    zero_idx = None

cbar2 = fig.colorbar(sm, ax=axes[1], orientation='horizontal', pad=0.01, fraction=0.05, aspect=30,
                     ticks=tick_locs)
cbar2.set_label("Anomaly (°C)", fontsize=12)

# Set tick labels
cbar2.set_ticklabels(tick_labels)

# Bold zero label to highlight center
if zero_idx is not None:
    cbar2.ax.xaxis.set_tick_params(labelsize=10)
    for tick_label in cbar2.ax.get_xticklabels():
        tick_label.set_fontsize(10)
    cbar2.ax.get_xticklabels()[zero_idx].set_fontweight('bold')

plt.tight_layout()
plt.savefig("26june2019_temperature_anomaly_maps_smooth.png", dpi=300, bbox_inches='tight')
plt.show()
